module net.BurtonRadons.dig.platform.spinner;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.canvas;
private import net.BurtonRadons.dig.platform.control;

//private import std.c.windows.windows;


/+
#ifdef DOXYGEN_MUST_NOT_SEE_THIS
+/

class SpinnerBox : Canvas
{
    private import net.BurtonRadons.dig.platform.base;

    this (Spinner parent)
    {
        super (parent);
        pad (0, 0);
        width (12);
        //height (textHeight ());
        onPaint.add (&digCommonDoPaint);
        onLButtonDown.add (&digCommonDoLButtonDown);
        onLButtonUp.add (&digCommonDoLButtonUp);
        onMouseMove.add (&digCommonDoMouseMove);
        onMouseLeave.add (&digCommonDoMouseLeave);
        sticky ("^v");
    }

    /** Set whether this is active (true) or grayed and inactive (false). */
    void enabled (bit value)
    {
        digCommonEnabled = value;
    }

    void step (float delta)
    {
        Spinner spinner = cast (Spinner) parent ();
        float value;

        value = delta * spinner.rangeStep () + spinner.value ();
        spinner.valueBound (value);
    }

    void digCommonDoLButtonDown (Event event)
    {
        if (!digCommonEnabled)
            return;

        digCommonMode = 0;
        if (event.y <= height () / 2)
            digCommonDown = 0, step (+1);
        else
            digCommonDown = 1, step (-1);
        captureMouse ();
        paint ();
        timer (400, &digCommonDoScrollAgain);
    }

    void digCommonDoMouseLeave ()
    {
        if (digCommonUpHighlit || digCommonDownHighlit)
        {
            digCommonUpHighlit = digCommonDownHighlit = false;
            paint ();
        }
    }

    int half ()
    {
        return ((height () - 0) + 1) / 2 + 0;
    }

    /* Handle swinging and moving to a swing. */
    void digCommonDoMouseMove (Event e)
    {
        /* Save the new values. */
        int nx = e.x;
        int ny = e.y;

        if (isCaptor ())
        {
            /* Switch to swing mode if the mouse is out of vertical range. */
            if (!digCommonMode && (ny < 0 || ny > height ()))
            {
                digCommonMode = 1;
                paint ();
            }

            /* Step the delta if in swing mode. */
            if (digCommonMode == 1)
                step (digCommonMY - ny);
        }

        /* Store the new values for button pushing. */
        digCommonMX = nx;
        digCommonMY = ny;

        if (digCommonMY < half ())
        {
            if (digCommonDownHighlit)
            {
                digCommonDownHighlit = false;
                digCommonUpHighlit = true;
                paint ();
            }
            else if (!digCommonUpHighlit)
            {
                digCommonUpHighlit = true;
                paint ();
            }
        }
        else
        {
            if (digCommonUpHighlit)
            {
                digCommonUpHighlit = false;
                digCommonDownHighlit = true;
                paint ();
            }
            else if (!digCommonDownHighlit)
            {
                digCommonDownHighlit = true;
                paint ();
            }
        }
    }

    void digCommonDoScrollAgain (Event e)
    {
        if (!isCaptor () || digCommonMode)
            return;

        if (digCommonMY < 0 || digCommonMY > height ())
            return;

        if (digCommonDown == 0)
        {
            if (digCommonMY >= height () / 2)
                goto retry;
            step (+1);
        }
        else
        {
            if (digCommonMY < height () / 2)
                goto retry;
            step (-1);
        }

    retry:
        timer (200, &digCommonDoScrollAgain);
    }

    void digCommonDoLButtonUp ()
    {
        if (!isCaptor ())
            return;
        digCommonDown = -1;
        digCommonMode = 0;
        releaseMouse ();
        paint ();
    }

    void digCommonDoPaint ()
    {
        beginPaint ();
        clear (backgroundColor ());
        spinnerDraw (0, 0, width (), height (), digCommonDown == 0 || digCommonMode, digCommonDown == 1 || digCommonMode, digCommonUpHighlit, digCommonDownHighlit);
        endPaint ();
    }
    
    int digCommonDown = -1;
    int digCommonMode = 0;
    int digCommonMX, digCommonMY;
    bit digCommonEnabled = true;
    bit digCommonUpHighlit;
    bit digCommonDownHighlit;
}

/+
#endif
+/

/** A spinner is an editor with a up/down button to the right of it and
  optionally a caption to the left.  The up/down button can be clicked 
  or pressed and swung to change the number in the edit box; the control
  in total is intended to be similar to 3DS Max's ubiquitous spinners.

  For example, the following code creates a spinner labeled "Length:"
  that allows controlling a number from 0 to 100 with 0.1 steps (that
  is to say, pressing the up/down button or swinging on it results in
  these steps per press and pixel respectively):

  @code

with (new Spinner (this))
{
    grid (0, 0); // Place the spinner in grid-fitting.
    caption ("Length:"); // Set the caption displayed on the left.
    range (0, 100, 0.1); // Set the range and steps.
}

  @endcode

  The primary dispatcher is the onChange message which is sent when the
  buttons change the number or when the user inputs a valid number in
  the edit box.  An out-of-range number there is clamped.

  */

class Spinner : Control
{
    private import net.BurtonRadons.dig.platform.editText;
    private import net.BurtonRadons.dig.platform.frame;
    private import net.BurtonRadons.dig.platform.label;
    private import net.BurtonRadons.dig.platform.windows;
    private const wchar [] digPlatformClassName = "digPlatformSpinner";
    
    /** Dispatcher for the "onChange" message for the Spinner control. 
      * The "add" method takes several delegate forms as input:
      *
      * @code
      * void delegate (Spinner spinner, float value);
      * void delegate (float value);
      * @endcode
      *
      * "spinner" is the spinner control that generates this message,
      * while "value" is the new setting.
      */

    struct ChangeDispatcher
    {
        /** A form of method call for the #ChangeDispatcher that takes two arguments. */
        alias void delegate (Spinner spinner, float value) aMethodA;

        /** A second form of method call that takes just one argument. */
        alias void delegate (float value) aMethodB;

        aMethodA [] methodas; /**< The list of first-form methods to call on notify. */
        aMethodB [] methodbs; /**< The list of second-form methods to call on #notify. */
        
        /** Add a method to the list that takes (Spinner spinner, float value) arguments. */
        void add (aMethodA method) { methodas ~= method; }

        /** Add a method to the list that takes (float value) arguments. */
        void add (aMethodB method) { methodbs ~= method; }

        /** Notify the methods that an event has occured. */
        void notify (Spinner spinner, float value)
        {
            for (aMethodA *m = methodas, n = m + methodas.length; m < n; m ++)
                (*m) (spinner, value);
            for (aMethodB *m = methodbs, n = m + methodbs.length; m < n; m ++)
                (*m) (value);
        }
    }

    ChangeDispatcher onChange;
    /**< Dispatched when the edit box has been modified or when the spin
       * box has been clicked or swung upon.  Events take
       * (Spinner spinner, float value) or (float value) arguments.
       */

    /** Setup the GUI class for windows. */
    static this ()
    {
        digPlatformWindowClass.style = 0;//CS_HREDRAW | CS_VREDRAW;
        digPlatformWindowClass.lpfnWndProc = &Frame.digPlatformWindowProc;
        digPlatformWindowClass.cbClsExtra = 0;
        digPlatformWindowClass.cbWndExtra = 0;
        digPlatformWindowClass.hInstance = hinst;
        digPlatformWindowClass.hCursor = LoadCursorA ((_HANDLE) 0, IDC_ARROW);
        digPlatformWindowClass.hbrBackground = (_HANDLE) 0;//(_HBRUSH) (backgroundColor + 1);
        digPlatformWindowClass.lpszMenuName = null;
        digPlatformWindowClass.lpszClassName = wtoStringz (digPlatformClassName);
        std.c.windows.windows.RegisterClassA (&digPlatformWindowClass);
    }

    /** Assign the parent, create the spinner and set it up. */
    this (Control parent)
    {
        super (parent);

        digPlatformStyle |= WS_CHILD | WS_VISIBLE;
        digPlatformHWNDCreate (0, digPlatformClassName, null, digPlatformStyle, (_HANDLE) 0);
        digPlatformPadX = digPlatformPadY = 0;
        digPlatformBorderX = digPlatformBorderY = 0;

        create ();
    }

    /** Set the caption displayed on the left of the spinner. */

    void caption (char [] text)
    {
        digPlatformCaption = text;
        create ();
    }

    /** Set the lower and upper bounds of the value and the 
      * step to use for the spin box.
      */

    void range (float lower, float upper, float step)
    {
        digPlatformLower = lower;
        digPlatformUpper = upper;
        digPlatformStep = step;
    }

    /** Get the step parameter. */

    float rangeStep ()
    {
        return digPlatformStep;
    }

    /** Set the current value.  This is not affected by range, so
      * out-of-bounds values will be silently accepted.  It does
      * not produce an onChange message.
      */

    void value (float value)
    {
        digPlatformIgnore = true;
        digPlatformEditor.text (fmt ("%g", value));
        digPlatformCurrent = value;
        digPlatformIgnore = false;
    }

    /** Get the current value. */

    float value ()
    {
        return digPlatformCurrent;
    }

    /** Set the value after bounding it, and send a onChange message. */

    void valueBound (float value)
    {
        value = fmid (digPlatformLower, value, digPlatformUpper);
        if (digPlatformCurrent == value)
            return;
        this.value (value);
        onChange.notify (this, value);
    }

    /** Set whether this is active (true) or grayed and inactive (false). */
    void enabled (bit value)
    {
        if (digPlatformLabel !== null)
            digPlatformLabel.enabled (value);
        if (digPlatformEditor !== null)
            digPlatformEditor.enabled (value);
        if (digPlatformSpinnerBox !== null)
            digPlatformSpinnerBox.enabled (value);
    }

    /** Create the sub-items of the control. */

    void create ()
    {
        if (digPlatformCaption !== null)
        {
            if (digPlatformLabel === null)
                digPlatformLabel = new Label (this);
            digPlatformLabel.grid (0, 0);
            digPlatformLabel.caption (digPlatformCaption);
        }
        else if (digPlatformLabel !== null)
        {
            delete digPlatformLabel;
            digPlatformLabel = null;
        }

        if (digPlatformEditor === null)
            with (digPlatformEditor = new EditText (this))
            {
                width (50);
                onChange.add (&digPlatformEditorChanged);
            }

        if (digPlatformSpinnerBox === null)
            with (digPlatformSpinnerBox = new SpinnerBox (this))
            {
            }

        digPlatformEditor.grid (digPlatformCaption !== null ? 1 : 0, 0);
        digPlatformSpinnerBox.grid (digPlatformCaption !== null ? 2 : 1, 0);
    }

protected:
    static _WNDCLASS digPlatformWindowClass; /**< The window class used by spinners. */
    EditText digPlatformEditor; /**< Link to the edit box part of the spinner. */
    char [] digPlatformCaption; /**< The caption to put on the left. */
    Label digPlatformLabel; /**< Link to the caption object or null if none is used. */
    SpinnerBox digPlatformSpinnerBox; /**< Link to the up and down arrows. */

    float digPlatformLower; /**< The lower bound of values. */
    float digPlatformUpper; /**< The upper bound of values. */
    float digPlatformStep; /**< Individual steps for the up and down arrows. */
    float digPlatformCurrent; /**< The currently assigned value. */
    bit digPlatformIgnore = false; /**< Whether to avoid calling the #onChange dispatcher for now. */

    /** Notify #onChange if possible. */
    void digPlatformEditorChanged ()
    {
        float value;

        if (digPlatformIgnore)
            return;
        if (digPlatformEditor.getFloat (value))
        {
            value = fmid (digPlatformLower, value, digPlatformUpper);
            digPlatformCurrent = value;
            onChange.notify (this, value);
        }
    }

    /** Gridfit. */
    override void digPlatformChildMoved (Control control)
    {
        digPlatformGridfit ();
        digPlatformMoved ();
    }
}
